(function(_window) {
    'use strict';

    /**
     * Regex replacer for HTML escaping.
     *
     * @param {string} match matched character (must exist in HTML_CHARS).
     * @return {string} Html entity
     */
    function htmlReplacer(match) {
        return HTML_CHARS[match];
    }

    var uniqIdCounter = 0,
        HTML_CHARS = {
            '&': '&amp;',
            '<': '&lt;',
            '>': '&gt;',
            '"': '&quot;',
            "'": '&#x27;',
            '/': '&#x2F;',
            '`': '&#x60;'
        },
        HTML_UNESCAPE_CHARS = {
            '&lt;'  : '<',
            '&gt;'  : '>',
            '&amp;' : '&',
            '&#x2F;' : '/',
            '&nbsp;': ' '
        },
        reEscapedHtml = /&(?:amp|lt|gt|#x2F|nbsp|#\d+);/g,
        reHasEscapedHtml = RegExp(reEscapedHtml.source),
        utils = {
            /**
             * Get a unique id.
             *
             * @param {string} prefix
             */
            uniqueId: function(prefix) {
                var uniqId = ++uniqIdCounter;

                // Simply incrementing (as lodash does) won't work if we want to preserve id's
                // when we rebind (open save draft).
                return prefix + '-' + Math.ceil(Math.random() * 1000000) + '-' + uniqId;
            },

            /**
             * Return true if the value is an Integer.
             *
             * Uses ES6 function if defined.
             *
             * @param {*} val
             */
            isInteger: Number.isInteger || function(val) {
                return typeof val === "number" &&
                       isFinite(val) &&
                       Math.floor(val) === val;
            },

            /**
             * Functional utility to loop through an array calling a
             * callback each time to test for a condition to be meet, and
             * exit on first find.
             *
             * @param {Array} arr
             * @param {Function} cb
             * @returns {Boolean}
             */
            findFirst: function(arr, cb) {
                var i;

                for(i=0; i < arr.length; i++) {
                    if (cb(arr[i])) {
                        return true;
                    }
                }

                return false;
            },

            /**
             * For each capable of handling HtmlCollections and NodeLists.
             *
             * Note: use {@link Array#forEach} if you can.  This method is only useful
             * for collections that do not contain the forEach method on their prototype.
             *
             * @param {HtmlCollection|NodeList} collection
             * @param {function} cb
             */
            forEach: function (collection, cb) {
                for (var i = 0; i < collection.length; i++) {
                    /**
                     * @callback cb
                     * @param {*} value
                     * @param {*} index
                     */
                    cb.call(this, collection[i], i);
                }
            },


            /**
             * Get the value of a key within an object.
             * Path to the value is specified by an array of keys.
             * If value is not found, defaultValue will be returned.
             *
             * @param {object} o
             * @param {array} path
             * @param {*} defaultValue
             *
             * @return {*} value at path if found, else default value if provided, else null
             */
            get: function(o, path, defaultValue) {
                // TODO check if o is an object
                var i,
                    p = Array.isArray(path) ? path : [], // TODO allow conversion from 'array-like' types and from dot-notation string
                    l = p.length;

                for (i = 0; o != null && i < l; i++) {
                    o = o[p[i]];
                }

                return o != null ? o : defaultValue;
            },

            // Port from underscore
            debounce: function(func, wait, immediate) {
                var timeout;
                return function() {
                    var context = this, args = arguments;
                    var later = function() {
                        timeout = null;
                        if (!immediate) {
                            func.apply(context, args);
                        }
                    };
                    var callNow = immediate && !timeout;
                    clearTimeout(timeout);
                    timeout = setTimeout(later, wait);
                    if (callNow) {
                        func.apply(context, args);
                    }
                };
            },

            /**
             * Escape Html string
             *
             * String escaping meets OWASP recommendations.
             * This method is taken from YUI Escape API.
             * {@see http://yuilibrary.com/yui/docs/api/files/escape_js_escape.js.html}
             *
             * @param {string} html string
             * @return {string} escaped html string
             */
            escapeHtml: function(string) {
                return (string + '').replace(/[&<>"'\/`]/g, htmlReplacer);
            },

            /**
             * Unescape Html string
             *
             * This method is taken from lodash Unescape API.
             * {@see https://github.com/lodash/lodash/blob/master/unescape.js}
             *
             * @param {string} html string
             * @return {string} unescaped html string
             */
            unescapeHtml: function(html) {
                return (html && reHasEscapedHtml.test(html))
                    ? html.replace(reEscapedHtml, function(entity) {
                        return HTML_UNESCAPE_CHARS[entity] || String.fromCharCode(parseInt(entity.slice(2, -1), 10));
                    }) : html;
            },
            noop: function() { },

            /**
             * Fire a callback as soon as the DOMContentLoaded has been fired.
             *
             * Function is executed immediately if called after DOMContentLoaded
             * has already been fired.
             *
             * @param {function} cb callback function to execute after DOMContentLoaded
             */
            onDomContentLoaded: function(cb) {
                if (document.readyState === 'interactive' || document.readyState === 'complete') {
                    cb.call(this);
                } else {
                    document.addEventListener('DOMContentLoaded', function wrapper() {
                        cb.call(this);
                        document.removeEventListener('DOMContentLoaded', wrapper);
                    });
                }
            },

            /**
             * Remove an element from the DOM
             */
            removeElement: function(el) {
                if (el) {
                    if (!('remove' in Element.prototype)) {
                        if (el.parentNode) {
                            el.parentNode.removeChild(el);
                        }
                    } else {
                        el.remove();
                    }
                }
            },

            /**
             * Check if given element is descendant of a parent element
             */
            isDescendant: function(parent, child) {
                var node = child.parentNode;
                while (node) {
                    if (node == parent) {
                        return true;
                    }
                    node = node.parentNode;
                }
                return false;
            },

            /**
             * Check if given element is descendant of a given tag
             */
            isDescendantOfTag: function(parentTag, child) {
                var node = child.parentNode;
                while (node) {
                    if (node.tagName == parentTag) {
                        return true;
                    }
                    node = node.parentNode;
                }
                return false;
            },

            /**
             * Extracts the query/param from the url search string.
             * This is not required if our min sdk support is >= 5.0. We can directly use
             * UrlSearchParam api. This api is not available with Android 4.4 webview
             */
            getUrlSearchParam : function(queryString) {
                var queryMap, get;
                // If Webview supports URLSeaerchParam, use it
                if (typeof(URLSearchParams) != 'undefined') {
                    var urlParamMap = new URLSearchParams(queryString);
                    queryMap = urlParamMap;
                    get = function(key) {
                        return urlParamMap.get(key);
                    }
                } else {
                    // Android 4.4 and lower will fall to this implementation
                    var query = decodeURIComponent(queryString), urlParamMap = {};
                    if (query.charAt(0) == "?") {
                        query = query.substring(1);
                    }
                    var entries = query.split("&");

                    entries.forEach(function(entry){
                        var tokens = entry.split("=");
                        if (tokens && tokens.length == 2) {
                            urlParamMap[tokens[0]] = tokens[1];
                        }
                    });

                    queryMap = urlParamMap;
                    get = function(key) {
                        return urlParamMap[key];
                    }
                }

                return { queryMap : queryMap,
                         get : get
                }
            },


            /** Extracts the form parameter from form element
             * Android 4.4 webview doesn't have most of functions of FormData.
             * This function takes care if it's not available - by manually iterating over all.
             * If the apis are available, then it uses the native api.
             */
            getFormParams : function(element, paramKeyTag, paramValueTag) {
                var params = [];
                // FormData.keys() is not available in 4.4 webview
                if (FormData.prototype.keys) {
                    // Using native api
                    var fd = new FormData(element);
                    var keyList = fd.keys();
                    for (var keyEntry; !(keyEntry = keyList.next()).done;) {
                        var param = {};
                        param[paramKeyTag] = keyEntry.value;
                        param[paramValueTag] = fd.get(keyEntry.value);
                        params.push(param);
                    }
                } else {
                    var id, allElements = element.getElementsByTagName("*");
                    for (id = 0; id < allElements.length; id++) {
                        if (allElements[id].name) {
                            var param = {};
                            param[paramKeyTag] = allElements[id].name;
                            param[paramValueTag] = allElements[id].value;
                            params.push(param);
                        }
                    }
                }
                return params;
            }
    };

    _window.utils = utils;
})(window);

(function(_window) {
    _window.theMainWindow = {
        showLinkWarning: function(link) {
            ConversationInterface.handleUnsecureLink(link.href, link.text);
        },
        showPasswordWarning: function(link) {
        },
        showFormWarning: function(form, cb, preventFormSubmit) {
        },
    };
})(window);
